"""Framework classes for generation of ecp test cases."""
# Copyright The Mbed TLS Contributors
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import List

from . import test_data_generation
from . import bignum_common

class EcpTarget(test_data_generation.BaseTarget):
    #pylint: disable=abstract-method, too-few-public-methods
    """Target for ecp test case generation."""
    target_basename = 'test_suite_ecp.generated'

class EcpP192R1Raw(bignum_common.ModOperationCommon,
                   EcpTarget):
    """Test cases for ecp quasi_reduction()."""
    symbol = "-"
    test_function = "ecp_mod_p192_raw"
    test_name = "ecp_mod_p192_raw"
    input_style = "fixed"
    arity = 1

    moduli = ["fffffffffffffffffffffffffffffffeffffffffffffffff"] # type: List[str]

    input_values = [
        "0", "1",

        # Modulus - 1
        "fffffffffffffffffffffffffffffffefffffffffffffffe",

        # First 8 number generated by random.getrandbits(384) - seed(2,2)
        ("cf1822ffbc6887782b491044d5e341245c6e433715ba2bdd"
         "177219d30e7a269fd95bafc8f2a4d27bdcf4bb99f4bea973"),
        ("ffed9235288bc781ae66267594c9c9500925e4749b575bd1"
         "3653f8dd9b1f282e4067c3584ee207f8da94e3e8ab73738f"),
        ("ef8acd128b4f2fc15f3f57ebf30b94fa82523e86feac7eb7"
         "dc38f519b91751dacdbd47d364be8049a372db8f6e405d93"),
        ("e8624fab5186ee32ee8d7ee9770348a05d300cb90706a045"
         "defc044a09325626e6b58de744ab6cce80877b6f71e1f6d2"),
        ("2d3d854e061b90303b08c6e33c7295782d6c797f8f7d9b78"
         "2a1be9cd8697bbd0e2520e33e44c50556c71c4a66148a86f"),
        ("fec3f6b32e8d4b8a8f54f8ceacaab39e83844b40ffa9b9f1"
         "5c14bc4a829e07b0829a48d422fe99a22c70501e533c9135"),
        ("97eeab64ca2ce6bc5d3fd983c34c769fe89204e2e8168561"
         "867e5e15bc01bfce6a27e0dfcbf8754472154e76e4c11ab2"),
        ("bd143fa9b714210c665d7435c1066932f4767f26294365b2"
         "721dea3bf63f23d0dbe53fcafb2147df5ca495fa5a91c89b"),

        # Next 2 number generated by random.getrandbits(192)
        "47733e847d718d733ff98ff387c56473a7a83ee0761ebfd2",
        "cbd4d3e2d4dec9ef83f0be4e80371eb97f81375eecc1cb63"
    ]

    @property
    def arg_a(self) -> str:
        return super().format_arg('{:x}'.format(self.int_a)).zfill(2 * self.hex_digits)

    def result(self) -> List[str]:
        result = self.int_a % self.int_n
        return [self.format_result(result)]

    @property
    def is_valid(self) -> bool:
        return True
